import Mathlib.Analysis.SpecialFunctions.Gamma.Basic
import Mathlib.Analysis.SpecialFunctions.Trigonometric.Basic
import Mathlib.Data.Complex.Exponential

open Complex Real

namespace RiemannZeta

noncomputable def chi (s : ℂ) : ℂ :=
  2^s * π^(s - 1) * sin (π * s / 2) * Complex.gamma (1 - s)
open RiemannZeta

/-- Extensión analítica de la función zeta mediante la simetría funcional -/
noncomputable def zeta_extended (s : ℂ) : ℂ :=
  if h : s.re > 1 then
    zeta_series s
  else if s = 1 then
    0  -- Polo simple en s = 1 (puede reemplazarse por `none` si se prefiere opción parcial)
  else
    (chi (1 - s) / chi s) * zeta_series (1 - s)
/-- Para Re(s) > 1, la extensión analítica coincide con la serie original -/
lemma zeta_extended_eq_series (s : ℂ) (h : 1 < s.re) :
  zeta_extended s = zeta_series s := by
  unfold zeta_extended
  rw [dif_pos h]

/-- La función funcional `chi(s)` es no nula en ℂ \ {0, 1} -/
lemma chi_ne_zero (s : ℂ) (h : s ≠ 0 ∧ s ≠ 1) : chi s ≠ 0 := by
  unfold chi
  -- Probaremos que cada factor de chi(s) es distinto de cero
  apply mul_ne_zero
  · apply mul_ne_zero
    · exact Complex.exp_ne_zero (s * Complex.log 2)  -- 2^s ≠ 0
    · exact Complex.exp_ne_zero ((s - 1) * Complex.log π)  -- π^(s - 1) ≠ 0
  apply sin_pi_half_ne_zero
  intro contra
  rcases contra with ⟨n, hs⟩
  subst hs
  have : s = 0 := by ring
  exact h.1 this
  · have hγ : Complex.gamma (1 - s) ≠ 0 := by
      apply gamma_one_sub_ne_zero
      intro contra
      rcases contra with ⟨z, hz, hz_le⟩
      -- Si 1 - s = z ∈ ℤ≤0 ⇒ s = 1 - z ⇒ z ≥ 1 ⇒ contradice s ≠ 1
      have : s = 1 - z := by rw [← hz]
      by_cases z = 0
      · subst z
        simp at this
        exact h.2 this
      · have : z ≥ 1 := by linarith
        contradiction
/-- `sin(πs/2) ≠ 0` cuando `s ∉ 2ℤ` -/
lemma sin_pi_half_ne_zero (s : ℂ) (h : ¬∃ n : ℤ, s = 2 * n) :
  sin (π * s / 2) ≠ 0 := by
  intro contra
  rw [Complex.sin_eq_zero_iff] at contra
  rcases contra with ⟨n, hπ⟩
  simp at hπ
  have : s = 2 * n := by
    field_simp at hπ
    linarith
  exact h ⟨n, this⟩
/-- `Gamma(1 - s) ≠ 0` cuando `1 - s ∉ ℤ≤0` -/
lemma gamma_one_sub_ne_zero (s : ℂ) (h : ¬∃ z : ℤ, 1 - s = z ∧ z ≤ 0) :
  Complex.gamma (1 - s) ≠ 0 := by
  apply Complex.Gamma_ne_zero
  intro contra
  rcases contra with ⟨z, hz, hz_neg⟩
  apply h
  use z
  exact ⟨hz.symm, hz_neg⟩

